/* --------------------------------------------------------------
  Validator.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Throw TypeError
 *
 * This method will throw the same message for every validation failure.
 *
 * @param {*} value Validated Value
 * @param {String} expectedType the Expected Type
 *
 * @private
 */
function _throw(value, expectedType) {
	throw new TypeError('Invalid value given, expected "' + expectedType + '" got "' + typeof value + '".', value);
}

export default {
	/**
	 * Validate Int Values
	 *
	 * @param {Number} value
	 *
	 * @throws TypeError If the argument is not valid.
	 */
	isInt(value) {
		if (value !== parseInt(value)) {
			_throw(value, 'integer');
		}
	},
	
	/**
	 * Validate Float Values
	 *
	 * @param {Number} value
	 *
	 * @throws TypeError If the argument is not valid.
	 */
	isFloat(value) {
		if (value !== Number(value) || value % 1 === 0) {
			_throw(value, 'float');
		}
	},
	
	/**
	 * Validate String Values
	 *
	 * @param {String} value
	 *
	 * @throws TypeError If the argument is not valid.
	 */
	isString(value) {
		if (value.constructor !== String) {
			_throw(value, 'string');
		}
	},
	
	/**
	 * Validate Bool Values
	 *
	 * @param {Boolean} value
	 *
	 * @throws TypeError If the argument is not valid.
	 */
	isBool(value) {
		if (value.constructor !== Boolean) {
			_throw(value, 'bool');
		}
	},
	
	/**
	 * Validate Object Values
	 *
	 * @param {Object} value
	 *
	 * @throws TypeError If the argument is not valid.
	 */
	isObject(value) {
		if (typeof value !== 'object') {
			_throw(value, 'object');
		}
	},
	
	/**
	 * Validate type of object.
	 *
	 * @param {String} type Expected object type name.
	 * @param {*} object The object to be validated.
	 *
	 * @throws TypeError If the argument is not valid.
	 */
	isTypeOf(type, object) {
		if (typeof object !== type) {
			_throw(object, type);
		}
	}
}
